// runlatex-sk.js for TeXLive.net
// Copyright 2020 2021, 2023 David Carlisle
// MIT Licence

// set here but local versions can be redefined after
// loading this file

var preincludes={};

var lltexts ={
    "Open in Overleaf": "Open in Overleaf",
    "TeXLive.net":      "LaTeX Online", // or "run latex" or whatever
    "Delete Output":    "Delete Output",
    "Compiling PDF":    "Compiling PDF",
    // The following not used on learnlatex.org
    "edit":             "edit",
    "copy":             "copy",
    "Added Code":       "Added code",
    "End Added Code":   "End Added code",
    "Top Caption":      "Edit and run this example:"
}

var lleditorlines=40;
var lladddefaultpreamble=true;
var lladddefaultengine=true;
var llusecaptions=true;

// debug by using https://httpbin.org/post
// set to null to omit from interface
var latexcgihost="https://texlive.net/cgi-bin/latexcgi";
var overleafhost=null; // "https://www.overleaf.com/docs";

var editors=[];

const noeditregex = /^\s*[/%#\*]+ *!TEX.*[^a-zA-Z]noedit *(\n|$)/i;
const norunregex = /^\s*([/%#\*]+ *!TEX.*[^a-zA-Z]none *|[^% \t\\][^\\]*)(\n|$)/i;
const commentregex = / %.*/;
const engineregex = /% *!TEX.*[^a-zA-Z](((pdf|xe|lua|u?p)?latex(-dev)?)|context|(pdf|xe|lua|u?p)?tex|make4ht) *\n/i;
const returnregex = /% *!TEX.*[^a-zA-Z](pdfjs|pdf|log) *\n/i;
const makeindexregex = /% *!TEX.*[^a-zA-Z]makeindex( [a-z0-9\.\- ]*)\n/ig;

var packageregex = [
    [ /\\includegraphics/,                    "\\usepackage[demo]{graphicx}\n"],
    [ /\\begin{equation|align|gather|flalign/,"\\usepackage{amsmath}\n"       ],
    [ /tikz|pgf/,                             "\\usepackage{tikz}\n"          ],
    [ /fancy/,                                "\\usepackage{fancyhdr}\n"      ],
    [ /addplot|axis/,                         "\\usepackage{pgfplots}\n"      ],
    [ /hyper|href|bookmark|\\url/,            "\\usepackage{hyperref}\n"      ],
    [ /\\newcolumntype/,                      "\\usepackage{array}\n"         ],
    [ /listing/,                              "\\usepackage{listings}\n"      ],
    [ /\\blind/,                              "\\usepackage{blindtext}\n"     ],
    [ /\\lipsum/,                             "\\usepackage{lipsum}\n"        ],
    [ /color/,                                "\\usepackage{xcolor}\n"        ],
    [ /pspicture/,                            "\\usepackage{pstricks}\n"      ]
];

function llexamples() {
    var p = document.getElementsByTagName("pre");
    var editor;
    var acemode;
    for(var i=0;i<p.length;i++) {
	    var pretext=p[i].innerText;
	    var grandfatherclass=p[i].parentElement.parentElement.getAttribute("class");
	    if(grandfatherclass==="highlight-acetex notranslate") {
	        acemode="ace/mode/latex";
        }
	    // class=noedit on pre or {: .class :} after closing ``` in markdown
	    if(!pretext.match(noeditregex) && !p[i].classList.contains('noedit') && grandfatherclass==="highlight-acetex notranslate") {
	        p[i].setAttribute("id","pre" + i);
	        if((lladddefaultpreamble && pretext.match(norunregex)) ||
	           (!lladddefaultpreamble &&  p[i].textContent.indexOf("\\documentclass") == -1 && !pretext.match(engineregex)) ||
	           p[i].classList.contains('norun') || (pretext.split(/\r\n|\r|\n/).length <3)) {
		        if(pretext.match(norunregex)) {
		            acemode="ace/mode/text";
		        }
	        } else {
		        // caption
		        if(llusecaptions && lltexts["Top Caption"]) {
		            var cpt = document.createElement("div");
		            cpt.setAttribute("class",'title-desc');
		            cpt.innerHTML=lltexts["Top Caption"];
		            p[i].parentNode.insertBefore(cpt, p[i]);
		        }
		        // space
		        var s = document.createElement("div");
		        s.setAttribute("class",'ace-spacer');
		        p[i].parentNode.insertBefore(s, p[i].nextSibling);
		        // texlive.net
		        var r = document.createElement("a");
		        r.innerText=lltexts["TeXLive.net"];
		        r.setAttribute("class","sd-sphinx-override sd-btn sd-text-wrap sd-btn-outline-primary reference external");
		        r.setAttribute("onclick",'latexcgi("pre' + i + '")');
		        r.setAttribute("id","lo-pre" + i);
		        p[i].parentNode.insertBefore(r, p[i].nextSibling);
		        if(overleafhost){
		            // overleaf
		            var o = document.createElement("button");
		            o.innerText=lltexts["Open in Overleaf"];
		            o.setAttribute("class","sd-sphinx-override sd-btn sd-text-wrap sd-btn-outline-primary reference external");
		            o.setAttribute("onclick",'openinoverleaf("pre' + i + '")');
		            p[i].parentNode.insertBefore(o, p[i].nextSibling);
		            var f=document.createElement("span");
		            f.innerHTML="<form style=\"display:none\" id=\"form-pre" + i +
			            "\" action=\"" +
			            overleafhost +
			            "\" method=\"post\" target=\"_blank\"></form>";
		            p[i].parentNode.insertBefore(f, p[i].nextSibling);
		        }
		        var f2=document.createElement("span");
		        f2.innerHTML="<form style=\"display:none\" id=\"form2-pre" + i +
		            "\" name=\"form2-pre" + i +
		            "\" enctype=\"multipart/form-data\" action=\"" +
		            latexcgihost +
		            "\" method=\"post\" target=\"pre" + i +
		            "ifr\"></form>";
		        p[i].parentNode.insertBefore(f2, p[i].nextSibling);
	        }
	        p[i].textContent=pretext.replace(/\s+$/,'');
	        editor = ace.edit(p[i]);
	        ace.config.set('basePath', 'https://cdnjs.cloudflare.com/ajax/libs/ace/1.30.0') ;
            editor.setOptions({fontSize: "14px"});
 	        editor.getSession().setMode(acemode);
	        editor.setOption("minLines",1);
	        editor.setOption("maxLines",lleditorlines);
	        editor.setShowPrintMargin(false);
	        editor.setHighlightActiveLine(false);
	        editor.resize();
	        editors["pre" + i]=editor;
	    }
    }
}


function addinput(f,n,v) {
    var inp=document.createElement("input");
    inp.setAttribute("type","text");
    inp.setAttribute("name",n);
    inp.value =encodeURIComponent(v);
    f.appendChild(inp);
}

function addinputnoenc(f,n,v) {
    var inp=document.createElement("input");
    inp.setAttribute("type","text");
    inp.setAttribute("name",n);
    inp.value =v;
    f.appendChild(inp);
}

function addtextarea(f,n,v) {
    var inp=document.createElement("textarea");
    inp.setAttribute("type","text");
    inp.setAttribute("name",n);
    inp.textContent=v;
    f.appendChild(inp);
}

function openinoverleaf(nd) {
    var fm = document.getElementById('form-' + nd);
    fm.innerHTML="";
    var p = document.getElementById(nd);
    var t = editors[nd].getValue();

    var engv="pdflatex";
    var eng=t.match(engineregex);
    if(lladddefaultpreamble) {
	    if(t.indexOf("\\documentclass") == -1 && ( eng == null)) {
	        t=generatepreamble(t,editors[nd]);
	    }
    }
    if(eng != null) {
	    engv=eng[1].toLowerCase();

	    if(engv == "pdftex" || engv == "luatex" || engv == "xetex" || engv == "ptex" || engv == "uptex") {
	        t = "% Force main document for Overleaf\n\\let\\tmp\n\\documentclass\n" + t;
	    }
    }
    addinput(fm,"encoded_snip[]","\n" + t);
    addinput(fm,"snip_name[]","document.tex");
    if(typeof(preincludes) == "object") {
	    if(typeof(preincludes[nd]) == "object") {
	        var incl=preincludes[nd];
	        for(prop in incl) {
		        if(editors[prop]==null) {
		            addinput(fm,"encoded_snip[]",document.getElementById(prop).textContent);
		        } else {
		            addinput(fm,"encoded_snip[]",editors[prop].getValue());
		        }
		        addinput(fm,"snip_name[]",incl[prop]);
	        }
	    }
    }
    if(eng != null) {
	    if(engv.indexOf("platex") != -1 || engv.indexOf("ptex") != -1 || engv=="tex") {
	        addinput(fm,"encoded_snip[]","$latex = '" + engv + "';\n$bibtex = 'pbibtex';\n$dvipdf = 'dvipdfmx %O -o %D %S';");
	        addinput(fm,"snip_name[]","latexmkrc");
	        engv="latex_dvipdf";
	    } else if(engv == "pdftex" || engv == "luatex" || engv == "xetex") {
	        addinput(fm,"encoded_snip[]","$pdflatex = '" + engv + "';");
	        addinput(fm,"snip_name[]","latexmkrc");
	        engv="pdflatex";
	    }

    }
    addinput(fm,"engine",engv);
    fm.submit();
}

function copytoclipboard(nd){
    var p = document.getElementById(nd);
    var nn=document.createElement("textarea");
    nn.value=p.innerText;
    document.body.appendChild(nn);
    nn.select();
    document.execCommand("copy");
    document.body.removeChild(nn);
}


function allowedit(nd){
    var p = document.getElementById(nd);
    p.contentEditable="true";
    p.setAttribute("spellcheck","false");
    p.innerHTML=p.innerText;
    p.style.border="solid thin green";
}

function deleteoutput(nd){
    var b = document.getElementById('del-' + nd);
    var ifr = document.getElementById(nd + 'ifr');
    b.parentNode.removeChild(b);
    ifr.parentNode.removeChild(ifr);
}

function generatepreamble(t,e) {
    e.navigateFileStart();
    if(t.match(/koma|KOMA|addsec|\\scr|scrheadings/)){
        e.insert("\n% " + lltexts["Added Code"] + "\n\\documentclass{scrartcl}\n");
    } else {
	    e.insert("\n% " + lltexts["Added Code"] + "\n\\documentclass{article}\n");
    }
    for(var i=0;i<packageregex.length; i++){
	    if(t.match(packageregex[i][0])) e.insert(packageregex[i][1]);
    }
    e.insert("\n\\begin{document}\n% "  + lltexts["End Added Code"] + "\n\n");
    e.navigateFileEnd();
    e.insert("\n\n% " +
	         lltexts["Added Code"] +
	         "\n\\end{document}\n% "  +
	         lltexts["End Added Code"] +
	         "\n");
    return e.getValue();
}

function defaultengine(t) {
	if ((t.indexOf("\\usepackage{lua") !== -1) || (t.indexOf("\\directlua") !== -1) ){
	    return "lualatex";
	} else if (t.indexOf("fontspec") !== -1) {
	    return "xelatex";
	} else if (t.indexOf("pstricks") !==-1) {
	    return "latex";
	} else return "pdflatex";
}

function latexcgi(nd) {
    var fm = document.getElementById('form2-' + nd);
    fm.innerHTML="";
    var p = document.getElementById(nd);
    var t = editors[nd].getValue();
    var engv="pdflatex";
    var eng=t.match(engineregex);
    if(lladddefaultpreamble) {
	    if(t.indexOf("\\documentclass") == -1 && ( eng == null)) {
	        t=generatepreamble(t,editors[nd]);
	    }
    }
    addtextarea(fm,"filecontents[]",t);
    addinputnoenc(fm,"filename[]","document.tex");
    if(typeof(preincludes) == "object") {
	    if(typeof(preincludes[nd]) == "object") {
	        var incl=preincludes[nd];
	        for(prop in incl) {
		        if(editors[prop]==null) {
		            addtextarea(fm,"filecontents[]",document.getElementById(prop).textContent);
		        } else {
		            addtextarea(fm,"filecontents[]",editors[prop].getValue());
		        }
		        addinputnoenc(fm,"filename[]",incl[prop]);
	        }
	    }
    }
    if(eng != null) {
	    engv=eng[1].toLowerCase();
    } else if(lladddefaultengine) {
	    engv=defaultengine(t);
    }
    addinput(fm,"engine",engv);
    var rtn = t.match(returnregex);
    var rtnv = "";
    if(rtn == null) {
	    // ES6 / IE
	    if (typeof Symbol == "undefined") addinput(fm,"return","pdf");
    } else {
	    rtnv=rtn[1].toLowerCase();
	    addinput(fm,"return",rtnv);
    }
    var mki = makeindexregex.exec(t);
    while (mki != null) {
	    addinputnoenc(fm,"makeindex[]",mki[1]);
	    mki = makeindexregex.exec(t);
    }
    var b = document.getElementById('lo-' + nd);
    var ifr= document.getElementById(nd + "ifr");
    if(ifr == null) {
	    ifr=document.createElement("iframe");
	    ifr.setAttribute("width","100%");
	    ifr.setAttribute("height","500em");
	    ifr.setAttribute("id",nd + "ifr");
	    ifr.setAttribute("name",nd + "ifr");
	    p.parentNode.insertBefore(ifr, b.nextSibling);
	    d=document.createElement("button");
	    d.innerText=lltexts["Delete Output"];
		d.setAttribute("class","sd-sphinx-override sd-btn sd-text-wrap sd-btn-outline-primary reference external");
	    d.setAttribute("id","del-" + nd);
	    d.setAttribute("onclick",'deleteoutput("' + nd + '")');
	    p.parentNode.insertBefore(d, b.nextSibling);
    }
    var  loading=document.createElement("div");
    loading.id=nd+"load";
    loading.textContent=lltexts["Compiling PDF"] + " . . .";
    p.parentNode.insertBefore(loading, ifr);
    // scroll only if really close to the bottom
    var rect = b.getBoundingClientRect();
    if(document.documentElement.clientHeight - rect.bottom < 50){
	    window.scrollBy(0,150);
    }
    setTimeout(function () {
	    p.parentNode.removeChild(document.getElementById(nd+"load"));
    }, 1000);
    fm.submit();
}

// Ace editor theme depending on the PyData dark/light theme
function aceTheme() {
    var p = document.getElementsByTagName("pre");
    var editor;
    const dark = document.documentElement.dataset.theme == 'dark';
    themeName = dark ? "dracula" : "dawn";
    for(var i=0;i<p.length;i++) {
	    var grandfatherclass=p[i].parentElement.parentElement.getAttribute("class");
	    if(!p[i].classList.contains('noedit') && grandfatherclass==="highlight-acetex notranslate") {
	        editor = ace.edit(p[i]);
            editor.setTheme(`ace/theme/${themeName}`);
        }
    }
}

function onChangePyDataThemeAceTheme() {
    var observer = new MutationObserver(function(mutations) {
        aceTheme();
    })
    observer.observe(document.documentElement, {attributes: true, attributeFilter: ['data-theme']});
}

function onLoadAceTheme() {
    aceTheme();
}

window.addEventListener('load', llexamples, false);
window.addEventListener('load',onChangePyDataThemeAceTheme, false);
window.onload = onLoadAceTheme;
