/*
** Datei: DVICMD.C
** Autor: Ingo Eichenseher
**        Gerhard Wilhelms
**
** Aenderung: 12.11.92
*/

#include <stdio.h>
#include <setjmp.h>
#include <string.h>
#include <ctype.h>
#include <math.h>
#ifdef __TURBOC__
#ifdef IBMPC
#include <conio.h>
#else
#include <ext.h>
#endif
#endif
#include <stdarg.h>
#include <stdlib.h>
#ifndef UNIX
#include <stdlib.h>
#endif
#include "dvimisc.h"
#include "dvi.h"
#include "dvidvi.h"
#include "dviframe.h"
#include "dvihdcp.h"
#include "dvisplin.h"
#include "dvidraw.h"

#define TERM_LEN 78

#if defined(IBMPC) && defined(__TURBOC__)
unsigned _stklen = 16U * 1024U;
#endif

static jmp_buf      halt_jump;
static FILE        *input_file=NULL;
static char        *path;
static char         input_name[128]="dvi.opt";
static long         lower[10], upper[10];

static int (*out_rout[])(void) =
{
    screen,
    p6low,
    p6high,
    p6mid,
    fx80,
    shipfile,
    hplj,
    hpljlow,
    bj300,
    NULL
};


void out_newline(void)
{
    putchar('\n');
}

void out_string(char *s)
{
    fputs(s,stdout);
    fflush(stdout);
}

void halt(char *format, ... )
{
    extern void vprint(char *, va_list);
    va_list l;
    va_start(l,format);
    vprint(format,l);
    va_end(l);
    dvi_clean();
    longjmp(halt_jump,1);
}

int wait_for_sheet(void)
{
    int c;
    printf("Insert new sheet into printer\n");
    printf("Continue (y/n) ? ");
    c=getch(); c=tolower(c); putchar('\n');
#if UNIX
    while(getchar()!='\n');
#endif
    return c=='n';
}

static int get_onoff(char *string)
{
    if (!strcmp(string,"on")) return 1;
    else if (!strcmp(string,"off")) return 0;
    else halt("'on' or 'off' expected");
    return 0;
}

static void get_pages(char *string, int *first, int *last, int *step)
{
    char *s;
    *step=1; *first=1; *last=9999;
    if (!*string) return;
    if (isdigit(*string)) *first=(int)strtol(string,&s,10);
    else s=string;
    if (*s==':' || *s=='-')
    {
	s++;
	if (isdigit(*s))
	{
	    *last=(int)strtol(string=s,&s,10);
	    if (*s++==':')
		if (isdigit(*s))
		    *step=(int)strtol(string=s,&s,10);
	}
	else *last=9999;
    }
    else *last=*first;
    if (*step==0) *step=1;
    if (*first<*last && *step<0) *step = -*step;
    if (*last<*first && *step>0) *step = -*step;
}

static void get_dimen(char *string, double *dimen)
{
    char *s=string, c;
    double r;
    while(isdigit(*s) || *s=='.') s++;
    if (s==string)
    {
	print("Illegal dimension");
	return;
    }
    strlwr(s); c = *s; *s=0; r=atof(string); *s=c;
    if (r==0.0 && *s=='\0') *dimen=0.0;
    else if (!strcmp(s,"in")) *dimen=r;
    else if (!strcmp(s,"cm")) *dimen=r/2.54;
    else if (!strcmp(s,"pt")) *dimen=r/72.27;
    else if (!strcmp(s,"pc")) *dimen=r*12.0/72.27;
    else if (!strcmp(s,"bp")) *dimen=r/72.0;
    else if (!strcmp(s,"mm")) *dimen=r/25.4;
    else if (!strcmp(s,"dd")) *dimen=r*1238.0/72.27;
    else if (!strcmp(s,"cc")) *dimen=r*12*1238.0/72.27;
    else if (!strcmp(s,"sp")) *dimen=r/72.27/65536.0;
    else
    {
	print("Illegal dimension");
	return;
    }
}

static int is_default(long l[10])
{
    int i;
    for (i=0; i<10; i++)
	if (l[i]!=ASTERISK) return 0;
    return 1;
}

static int get_line(char *line, int file_only)
{
    if (input_file!=NULL)
    {
	if (fgets(line,128-1,input_file)==NULL)
	{
	    fclose(input_file);
	    input_file=NULL;
	    return get_line(line,file_only);
	}
	else
	{
	    char *p;
	    line[127]='\0';
	    if ((p=strrchr(line,'\n'))!=NULL) *p='\0';
	    /* if (*line) print("dvi>%s",line); */
	}
    }
    else
    {
	if (file_only) return 0;
	printnnl("dvi>");
	gets(line);
	term_clean();
    }
    return 1;
}

typedef enum
{
    cmd_not_found=-2, cmd_ambiguous=-1,
    cmd_hresolution, cmd_pkpath, cmd_width, cmd_height,
    cmd_options, cmd_dvifile, cmd_format, cmd_exit,
    cmd_output, cmd_vresolution, cmd_hoffset, cmd_voffset,
    cmd_hspread, cmd_vspread, cmd_eject, cmd_separate,
    cmd_lower, cmd_upper, cmd_input, cmd_magnification,
    cmd_thinout, cmd_path, cmd_landscape, cmd_logfile,
    cmd_hmargin, cmd_vmargin, cmd_copies, cmd_page,
    cmd_pictures, cmd_imgpath, cmd_density, cmd_vfpath,
    cmd_grpath, cmd_showfonts, cmd_singlesheet, cmd_memory,
    cmd_tracemem, cmd_tracechars, cmd_tfmpath,
    cmd_redirect, cmd_pixmem, cmd_maxmem, cmd_pathmem,
    cmd_port
} cmdnum_t;

typedef struct
{
    char *cmd_name;
    int cmd_number;
} cmd_t;

static cmd_t cmd_list[] =
{
    { "hresolution",    cmd_hresolution },
    { "pkpath",         cmd_pkpath },
    { "tfmpath",        cmd_tfmpath },
    { "width",          cmd_width },
    { "height",         cmd_height },
    { "options",        cmd_options },
    { "dvifile",        cmd_dvifile },
    { "format",         cmd_format },
    { "exit",           cmd_exit },
    { "output",         cmd_output },
    { "vresolution",    cmd_vresolution },
    { "hoffset",        cmd_hoffset },
    { "voffset",        cmd_voffset },
    { "hspread",        cmd_hspread },
    { "vspread",        cmd_vspread },
    { "eject",          cmd_eject },
    { "separate",       cmd_separate },
    { "lower",          cmd_lower },
    { "upper",          cmd_upper },
    { "input",          cmd_input },
    { "magnification",  cmd_magnification },
    { "thinout",        cmd_thinout },
    { "path",           cmd_path },
    { "landscape",      cmd_landscape },
    { "logfile",        cmd_logfile },
    { "hmargin",        cmd_hmargin },
    { "vmargin",        cmd_vmargin },
    { "copies",         cmd_copies },
    { "page",           cmd_page },
    { "pictures",       cmd_pictures },
    { "imgpath",        cmd_imgpath },
    { "density",        cmd_density },
    { "vfpath",         cmd_vfpath },
    { "grpath",         cmd_grpath },
    { "showfonts",      cmd_showfonts },
    { "singlesheet",    cmd_singlesheet },
    { "memory",         cmd_memory },
    { "tracemem",       cmd_tracemem },
    { "tracechars",     cmd_tracechars },
    { "redirect",       cmd_redirect },
    { "pixmem",         cmd_pixmem },
    { "maxmem",         cmd_maxmem },
    { "pathmem",        cmd_pathmem },
    { "port",           cmd_port },
    { NULL,             cmd_not_found }
};


static cmd_t out_list[]=
{
    { "screen",     0 },
    { "p6low",      1 },
    { "p6high",     2 },
    { "p6mid",      3 },
    { "fx80",       4 },
    { "file",       5 },
    { "hphigh",     6 },
    { "hplow",      7 },
    { "bj300",      8 },
    { "null",       9 },
    { NULL,        -1 }
};

static cmdnum_t cmp_cmd(char *cmd, cmd_t *cmd_list)
{
    int ret = cmd_not_found;

    strlwr(cmd);
    for (; cmd_list->cmd_name!=NULL; cmd_list++)
    {
	if (!strncmp(cmd,cmd_list->cmd_name, strlen(cmd)))
	{
	    if (ret!=cmd_not_found) return cmd_ambiguous;
	    else ret = cmd_list->cmd_number;
	}
    }
    return ret;
}

static char *cmd_name(cmdnum_t n)
{
    cmd_t *p;

    for (p=cmd_list; p->cmd_name!=NULL && p->cmd_number!=n; p++);
    if (p->cmd_name==NULL) return "";
    else return p->cmd_name;
}

static void get_output(char *arg)
{
    int i;

    if ((i=cmp_cmd(arg,out_list))<=cmd_ambiguous)
	print("Unknown or ambiguous output device");
    else
	shipout = out_rout[i];
}

static char *onoff(int v)
{
    return v ? "on":"off";
}

static void do_commands(int file_only)
{
    char line[128], *cmd, *arg;
    FILE *opt_file;
    long r;
    int i, k;

    while(1)
    {
	if (!get_line(line,file_only)) return;
	cmd=strtok(line," =\t\n");
	if (cmd==NULL) continue;
	if (*cmd=='\0') continue;
	arg=strtok((char*)NULL," =\t\n");

	switch(i=cmp_cmd(cmd,cmd_list))
	{
	    case cmd_ambiguous:
		print("Ambiguous command name");
		break;

	    case cmd_hresolution :
		if (arg) r=atol(arg);
		if (r<=0) print("Illegal resolution");
		else op.hres=r;
		break;

#ifdef IBMPC
	    case cmd_port :
		if (arg) r=atol(arg);
		if (r<0 || r>=4) print("Illegal port number");
		else op.biosdev=(int)r;
		break;
#endif

	    case cmd_pkpath :
		if (arg!=NULL) strcpy(op.pk_path,arg);
		else *op.pk_path=0;
		break;

	    case cmd_tfmpath :
		if (arg!=NULL) strcpy(op.tfm_path,arg);
		else *op.tfm_path=0;
		break;

	    case cmd_width :
		if (arg==NULL) op.width=0.0;
		else get_dimen(arg,&op.width);
		break;

	    case cmd_height :
		if (arg==NULL) op.height=0.0;
		else get_dimen(arg,&op.height);
		break;

	    case cmd_options :
		if (arg)
		{
		    catfe(input_name,arg,"opt");
		    opt_file=fopen(input_name,"w");
		}
		else opt_file=stdout;
		if (opt_file==NULL)
		{
		    print("Cannot open file %s",arg);
		    break;
		}
		fprintf(opt_file,"%s = %ld\n",
		    cmd_name(cmd_hresolution),op.hres);
		fprintf(opt_file,"%s = %ld\n",
		    cmd_name(cmd_vresolution),op.vres);
		if (op.new_mag)
		    fprintf(opt_file,"%s = %ld\n",
			cmd_name(cmd_magnification),op.new_mag);
		if (*op.pk_path)
		    fprintf(opt_file,"%s = %s\n",
			cmd_name(cmd_pkpath),op.pk_path);
		if (*op.tfm_path)
		    fprintf(opt_file,"%s = %s\n",
			cmd_name(cmd_tfmpath),op.tfm_path);
		if (*op.dvi_path)
		    fprintf(opt_file,"%s = %s\n",
			cmd_name(cmd_path),op.dvi_path);
		if (*op.img_path)
		    fprintf(opt_file,"%s = %s\n",
			cmd_name(cmd_imgpath),op.img_path);
		if (*op.input_path)
		    fprintf(opt_file,"%s = %s\n",
			cmd_name(cmd_grpath),op.input_path);
		if (*op.vf_path)
		    fprintf(opt_file,"%s = %s\n",
			cmd_name(cmd_vfpath),op.vf_path);
		if (!is_default(lower))
		    fprintf(opt_file,"%s = %s\n",
			cmd_name(cmd_lower), fmt_page(lower));
		if (!is_default(upper))
		    fprintf(opt_file,"%s = %s\n",
			cmd_name(cmd_upper), fmt_page(upper));
		if (op.width)
		    fprintf(opt_file,"%s = %.3fin\n",
			cmd_name(cmd_width),op.width);
		if (op.height)
		    fprintf(opt_file,"%s = %.3fin\n",
			cmd_name(cmd_height),op.height);
		if (op.hoffset)
		    fprintf(opt_file,"%s = %.3fin\n",
			cmd_name(cmd_hoffset),op.hoffset);
		if (op.voffset)
		    fprintf(opt_file,"%s = %.3fin\n",
			cmd_name(cmd_voffset),op.voffset);
		if (op.hmargin)
		    fprintf(opt_file,"%s = %.3fin\n",
			cmd_name(cmd_hmargin),op.hmargin);
		if (op.vmargin)
		    fprintf(opt_file,"%s = %.3fin\n",
			cmd_name(cmd_vmargin),op.vmargin);
		if (op.hspread)
		    fprintf(opt_file,"%s = %.3fin\n",
			cmd_name(cmd_hspread),op.hspread);
		if (op.vspread)
		    fprintf(opt_file,"%s = %.3fin\n",
			cmd_name(cmd_vspread),op.vspread);
		fprintf(opt_file,"%s = %s\n",
		    cmd_name(cmd_eject), onoff(op.formfeed));
		fprintf(opt_file,"%s = %s\n",
		    cmd_name(cmd_separate), onoff(op.separate));
		if (op.showfonts)
		    fprintf(opt_file,"%s = %s\n",
			cmd_name(cmd_showfonts), onoff(op.showfonts));
		if (op.landscape)
		    fprintf(opt_file,"%s = %s\n",
			cmd_name(cmd_landscape), onoff(op.landscape));
		if (op.copies!=1)
		    fprintf(opt_file,"%s = %d\n",
			cmd_name(cmd_copies),op.copies);
		fprintf(opt_file,"%s = %s\n",
		    cmd_name(cmd_pictures), onoff(op.show_img));
		fprintf(opt_file,"%s = %s\n",
		    cmd_name(cmd_memory), onoff(op.dvimemory));
		fprintf(opt_file,"%s = %s\n",
		    cmd_name(cmd_tracemem), onoff(op.tracemem));
		fprintf(opt_file,"%s = %s\n",
		    cmd_name(cmd_tracechars), onoff(op.tracechars));
		fprintf(opt_file,"%s = %g\n",
		    cmd_name(cmd_density),op.density);
		fprintf(opt_file,"%s = %s\n",
		    cmd_name(cmd_singlesheet), onoff(op.singlesheet));
		fprintf(opt_file,"%s = %s\n",
		    cmd_name(cmd_thinout), onoff(op.thin_out));
		for (k=0; out_rout[k]!=NULL; k++)
		    if (shipout==out_rout[k])
			fprintf(opt_file,"%s = %s\n",
			    cmd_name(cmd_output),out_list[k].cmd_name);
		if (op.pixmem)
		    fprintf(opt_file,"%s = %ld\n",
			cmd_name(cmd_pixmem),op.pixmem);
		if (op.maxmem)
		    fprintf(opt_file,"%s = %ld\n",
			cmd_name(cmd_maxmem),op.maxmem);
		if (op.pathmem)
		    fprintf(opt_file,"%s = %ld\n",
			cmd_name(cmd_pathmem),op.pathmem);
		if (*op.redirect)
		    fprintf(opt_file,"%s = %s\n",
			cmd_name(cmd_redirect),op.redirect);
#ifdef IBMPC
		if (op.biosdev!=0)
		    fprintf(opt_file,"%s = %d\n",
			cmd_name(cmd_port),op.biosdev);
#endif
		if (opt_file!=stdout) fclose(opt_file);
		break;

	    case cmd_dvifile :
		if (arg!=NULL) strcpy(dvi_name,arg);
		break;

	    case cmd_format :
		if (*dvi_name)
		{
		    int first, last, step;
		    if (arg==NULL)
		    {
			first=1; last=9999; step=1;
		    }
		    else get_pages(arg,&first,&last,&step);
		    format_pages(first,last,step);
		}
		else print("No filename specified");
		break;

	    case cmd_exit :
		return;

	    case cmd_output :
		if (arg) get_output(arg);
		break;

	    case cmd_vresolution :
		if (arg==NULL) break;
		r=atol(arg);
		if (r<=0) print("Illegal resolution");
		else op.vres=r;
		break;

	    case cmd_hoffset :
		if (arg==NULL) op.hoffset=0.0;
		else get_dimen(arg,&op.hoffset);
		break;

	    case cmd_voffset :
		if (arg==NULL) op.voffset=0.0;
		else get_dimen(arg,&op.voffset);
		break;

	    case cmd_hspread :
		if (arg==NULL) op.hspread=0.0;
		else get_dimen(arg,&op.hspread);
		break;

	    case cmd_vspread :
		if (arg==NULL) op.vspread=0.0;
		else get_dimen(arg,&op.vspread);
		break;

	    case cmd_eject :
		if (arg==NULL) print(onoff(op.formfeed));
		else op.formfeed = get_onoff(arg);
		break;

	    case cmd_separate :
		if (arg==NULL) print(onoff(op.separate));
		else
		{
		    op.separate = get_onoff(arg);
		    if (op.separate) op.thin_out = 0;
		}
		break;

	    case cmd_lower :
		if (arg==NULL) break;
		if (!get_bound(arg,lower))
		    print("Illegal page specification");
		break;

	    case cmd_upper :
		if (arg==NULL) break;
		if (!get_bound(arg,upper))
		    print("Illegal page specification");
		break;

	    case cmd_not_found : arg=cmd;
	    case cmd_input :
		if (arg==NULL) break;
		catfe(input_name,arg,"opt");
		input_file = fopene(input_name,"r",path,NULL);
		if (input_file==NULL)
		    print("Cannot open file %s",input_name);
		break;

	    case cmd_magnification :
		if (arg) r=atol(arg);
		else break;
		if (r<0) print("Illegal magnification");
		else op.new_mag=r;
		break;

	    case cmd_thinout :
		if (arg==NULL) print(onoff(op.thin_out));
		else
		{
		    op.thin_out = get_onoff(arg);
		    if (op.thin_out) op.separate = 0;
		}
		break;

	    case cmd_path :
		if (arg!=NULL) strcpy(op.dvi_path,arg);
		else *op.dvi_path=0;
		break;

	    case cmd_showfonts :
		if (arg==NULL) print(onoff(op.showfonts));
		else op.showfonts = get_onoff(arg);
		break;

	    case cmd_tracechars :
		if (arg==NULL) print(onoff(op.tracechars));
		else op.tracechars = get_onoff(arg);
		break;

	    case cmd_tracemem :
		if (arg==NULL) print(onoff(op.tracemem));
		else op.tracemem = get_onoff(arg);
		break;

	    case cmd_singlesheet :
		if (arg==NULL) print(onoff(op.singlesheet));
		else op.singlesheet = get_onoff(arg);
		break;

	    case cmd_memory :
		if (arg==NULL) print(onoff(op.dvimemory));
		else op.dvimemory = get_onoff(arg);
		break;

	    case cmd_pixmem :
		if (arg==NULL) print("pixmem=%ld",op.pixmem);
		else op.pixmem = atol(arg);
		break;

	    case cmd_maxmem :
		if (arg==NULL) print("maxmem=%ld",op.maxmem);
		else op.maxmem = atol(arg);
		break;

	    case cmd_pathmem :
		if (arg==NULL) print("pathmem=%ld",op.pathmem);
		else op.pathmem = atol(arg);
		break;

	    case cmd_landscape :
		if (arg==NULL) print(onoff(op.landscape));
		else op.landscape = get_onoff(arg);
		break;

	    case cmd_logfile :
		if (arg==NULL) op.log_name[0]='\0';
		else catfe(op.log_name,arg,"log");
		break;

	    case cmd_redirect :
		if (arg==NULL) op.redirect[0]='\0';
		else strcpy(op.redirect,arg);
		break;

	    case cmd_hmargin :
		if (arg==NULL) op.hmargin=0.0;
		else get_dimen(arg,&op.hmargin);
		break;

	    case cmd_vmargin :
		if (arg==NULL) op.vmargin=0.0;
		else get_dimen(arg,&op.vmargin);
		break;

	    case cmd_copies :
		if (arg==NULL) op.copies=1;
		else op.copies=atoi(arg);
		break;

	    case cmd_page :
		if (arg==NULL) prbyte(12);
		else
		{
		    int k=atoi(arg);
		    while(k-- > 0) prbyte(12);
		}
		break;

	    case cmd_pictures :
		if (arg==NULL) print(onoff(op.show_img));
		else op.show_img = get_onoff(arg);
		break;

	    case cmd_imgpath :
		if (arg!=NULL) strcpy(op.img_path,arg);
		else *op.img_path=0;
		break;

	    case cmd_density :
		if (arg==NULL) print("Density = %g",op.density);
		else
		{
		    op.density = atof(arg);
		    if (op.density < 0.0) op.density = 0.0;
		    if (op.density > 1.0) op.density = 1.0;
		}
		break;

	    case cmd_vfpath :
		if (arg!=NULL) strcpy(op.vf_path,arg);
		else *op.vf_path=0;
		break;

	    case cmd_grpath :
		if (arg!=NULL) strcpy(op.input_path,arg);
		else *op.input_path=0;
		break;
	}
	switch(i)
	{
	    case cmd_ambiguous:
	    case cmd_not_found:
	    case cmd_format:
	    case cmd_input:
		break;

	    default:
		dvi_clean();
	}
    }
}

static char *get_arg(int *argc, char ***argv)
{
    if ( (**argv)[2] ) return (**argv)+2;
    else if (*argc>1)
    {
	if ( *(*argv)[1]!='-' )
	{
	    (*argc)--;
	    return *++(*argv);
	}
	else return "";
    }
    else return "";
}

int main(int argc,char **argv)
{
    int interactive=1;
    int i, first, last, step;

    print( "This is DVI Version %s" , version_string );

    shipout = screen;
    path = getenv("PATH");

    install();
    if( !gr_install() )
        exit(1);

    for (i=0; i<10; i++) lower[i] = upper[i] = ASTERISK;
    for(argc--,argv++; argc; argc--, argv++)
    {
	if (**argv=='-')
	{
	    switch(tolower((*argv)[1]))
	    {
		case 'i' :
		    strcpy(input_name,get_arg(&argc,&argv));
		    break;
		case 'f' :
		    get_pages(get_arg(&argc,&argv),&first,&last,&step);
		    interactive=0;
		    break;
		case 'o' :
		    get_output(get_arg(&argc,&argv));
		    break;
		case 'l' :
		    strcpy(op.log_name,get_arg(&argc,&argv));
		    break;
	    }
	}
	else
	{
	    strcpy(dvi_name,*argv);
	}
    }


    if (*op.log_name)
    {
	catfe(op.log_name,op.log_name,"log");
	log_file = fopen(op.log_name,"w");
	if (log_file==NULL) print("Cannot open logfile");
    }

    if (*input_name)
    {
	input_file = fopen(input_name,"r");
	if (input_file==NULL)
	{
	    catfe(input_name,input_name,"opt");
	    if (path!=(char*)NULL)
		input_file = fopene(input_name,"r",path,(char*)NULL);
	    else
		input_file = fopen(input_name,"r");
	}
	if (input_file!=NULL)
	{
	    if (!setjmp(halt_jump)) do_commands(1);
	}
	else print("Cannot open input file %s",input_name);
    }

    if (interactive || *dvi_name=='\0')
    {
	setjmp(halt_jump);
	do_commands(0);
    }
    else
    {
	if (!setjmp(halt_jump)) format_pages(first,last,step);
	else print("Program aborted");
    }

    dvi_clean();
    if (log_file) fclose(log_file);
    if (input_file) fclose(input_file);
    close_missing();

    gr_destall();
    destall();

    mem_test();

    return 0;
}

