
IMPLEMENTATION MODULE DCLInterface;

(* Author:         Andrew Trevorrow
   Implementation: University of Hamburg Modula-2 under VAX/VMS version 4
   Date Started:   June, 1985

   Description:
   Gets the DVItoVDU command line and extracts the DVI file parameter
   and qualifier values according to DVITOVDU.CLD.
   Extraction is done using the VMS CLI routines.

   Revised:
   July, 1985 (to be consistent with TeX and IMPRINT from Kellerman & Smith)
 - /FONT_DIRECTORY now specifies a list of string values.
 - /XSIZE and /YSIZE are specified as dimensions with units in
   IN, CM, MM, PC, PT or PX (corresponding to the new DVItoVDU commands).
   Their values are converted to the nearest pixel and exported via
   paperwd and paperht.

   September, 1986
 - /FONT_DIRECTORY is back to a single value!
 - /DUMMY_FONT is assumed to reside in /FONT_DIRECTORY.

   November, 1987
 - Added /TFM_DIRECTORY qualifier.

   June--August, 1988 (while at Aston University)
 - Added /PSPREFIX qualifier.
 - Added /HOFFSET and /VOFFSET qualifiers.

   November, 1988 (Niel Kempson <tex@uk.ac.cranfield.rmcs>,
		  RMCS Shrivenham, SWINDON SN6 8LA, United Kingdom)
 - Updated to use VMS version 4.x system service calls
    i.e. use the new version of CLI$GET_VALUE, and SYS$TRNLNM instead
	 of SYS$TRNLOG.
 - Removed /FONT_DIRECTORY qualifier.
 - Added /PK_FONT_DIRECTORY & /PXL_FONT_DIRECTORY qualifiers.
 - Use the MODULA run time library function Parse to get the DVI file 
   specification. Delete the ExplicitExt procedure.

    December, 1988
  - Added /XORIGIN & /YORIGIN qualifiers to incorporate changes made by 
    Phil Taylor (CHAA006@UK.AC.RHBNC.VAXB).

   April, 1990 (Brian {Hamilton Kelly} <tex@uk.ac.cranfield.rmcs>,
		  RMCS Shrivenham, SWINDON SN6 8LA, United Kingdom)
 - Qualifiers which have a string value extracted both with and without logical
   name translation, to support use of search list logical names in qualifiers
   such as /PK_FONT_DIRECTORY
*)

  FROM SYSTEM IMPORT 
                     BYTE, SHORTWORD, ADR;

  FROM LNMDefinitions IMPORT 
                             LNM$V_CASE_BLIND,
                             LNM$_STRING;

  FROM CommandLanguageInterface IMPORT 
                                       CLI$PRESENT, CLI$GET_VALUE;

  FROM VMS IMPORT 
                  SYS$TRNLNM;

  FROM Conversions IMPORT 
                          StringToCard, StringToReal, Done, ShortToInt;

  FROM ScreenIO IMPORT 
                       Write, WriteString, WriteLn, Halt;

  FROM FileSystem IMPORT
		         Parse, ShowStatus;

  CONST
    NULL = 0C;             (* SYSDEP: terminates a string *)

  VAR
    value : stringvalue;   (* temporary string *)
    DiscardValue : stringvalue;	(* qualifiers not required dumped here *)

(******************************************************************************)

  PROCEDURE GetDVIFile;

(* Get DVI file name from command line *)

    VAR
      i, j, status : CARDINAL;
      fnstatus : BITSET;
      DefaultDVIFileSpec, ParsedDVIFileSpec, GivenSpec : stringvalue;
      RetLen : SHORTWORD;

  BEGIN
    DVIname := '';
    status := CLI$GET_VALUE('FILESPEC',DVIname,RetLen);
    (* CLD ensures it is there *)
    i := CARDINAL(ShortToInt(RetLen)) - 1;
    j := HIGH(DVIname);
    WHILE (j > i) DO
      DVIname[j] := NULL;
      DEC(j);
    END;

    IF DVIname[i] = ':' THEN                       (* translate logical name *)
      IF Translate(DVIname,i) THEN
      (* do nothing more in either case *)
      END;
    ELSE 
      (* 
	By default, use the latest version of file type .DVI
      *)
      DefaultDVIFileSpec := '';
      ParsedDVIFileSpec := '';
      Append(DefaultDVIFileSpec, '.DVI;0');
      Parse(DVIname, DefaultDVIFileSpec, ParsedDVIFileSpec, fnstatus);
      (*
	Copy the parsed file specification back into DVIname.
      *)
      DVIname := '';
      Append(DVIname, ParsedDVIFileSpec);
    END;
(* bad DVIname will be detected upon open in main module *)
  END GetDVIFile;

(******************************************************************************)

  PROCEDURE Translate (VAR logname : ARRAY OF CHAR;       (* in/out *)
                         lastpos : CARDINAL)            (* position of colon *)
                    : BOOLEAN;

(* SYSDEP: lastpos in logname should be a colon.
   Return TRUE if given logname can be translated and return the
   equivalence name in logname.
   If no translation, return FALSE and don't alter logname.
*)

    VAR
      i, status, ReturnLength : CARDINAL;
      attributes : BITSET;
      tablename : stringvalue;
      itemlist : ARRAY [0..1] OF VMSItemList;
      itemlistptr : VMSItemListPtr;

  BEGIN
    logname[lastpos] := NULL;  (* remove colon from logical name *)

    attributes := {LNM$V_CASE_BLIND};(* case insentitive name matching *)
    tablename[0] := NULL;
    Append(tablename, 'LNM$DCL_LOGICAL');
    ReturnLength := 0;

    WITH itemlist[0] DO
      BufferLength := SHORTWORD(HIGH(logname));
      ItemCode := SHORTWORD(LNM$_STRING);
      BufferAddress := ADR(logname[0]);
      RetLenAddress := ADR(ReturnLength);
    END;
    WITH itemlist[1] DO
      BufferLength := SHORTWORD(0);
      ItemCode := SHORTWORD(0);
      BufferAddress := NIL;
      RetLenAddress := NIL;
    END;
    itemlistptr := ADR(itemlist[0]);

(* DEBUG
    WriteString('Translating logical name "');
    WriteString(logname);
    WriteString('".');
    WriteLn;

    WriteString('Using logical name table "');
    WriteString(tablename);
    WriteString('".');
    WriteLn;
GUBED *)

    status := SYS$TRNLNM(attributes,tablename,logname,
                         		     0,itemlistptr);

    IF ODD(status) THEN
      FOR i := ReturnLength TO lastpos DO
        logname[i] := NULL;                     (* SYSDEP: pad with NULLs *)
      END;

(* DEBUG
	WriteString('Translated to "');
	WriteString(logname);
	WriteString('".');
	WriteLn;
GUBED *)

      lastpos  := ReturnLength;
      ReturnLength := 0;

      WHILE ODD(SYS$TRNLNM(attributes,tablename,logname,0,itemlistptr)) DO
        	
        FOR i := ReturnLength TO lastpos DO
          	    logname[i] := NULL;
          	
        END;

(* DEBUG 
	WriteString('Translated to "');
	WriteString(logname);
	WriteString('".');
	WriteLn;
GUBED *)

        lastpos := ReturnLength;
        	ReturnLength := 0;
      END;
      RETURN TRUE;
    ELSE                       (* no initial translation; restore colon *)
      logname[lastpos] := ':';
      RETURN FALSE;
    END;
  END Translate;

(******************************************************************************)

  PROCEDURE GetCardinal (qualifier : ARRAY OF CHAR;
                         VAR n : CARDINAL);

(* Check if qualifier is present.  If so, then get value, check it is
   a positive integer, and return via n.
*)

    VAR
      i, j, status : CARDINAL;
      RetLen : SHORTWORD;

  BEGIN
    IF ODD(CLI$PRESENT(qualifier)) THEN
      (* CLD ensures it has a value *)
      status := CLI$GET_VALUE(qualifier,value,RetLen);
      i := CARDINAL(ShortToInt(RetLen)) - 1;
      j := HIGH(value);
      WHILE (j > i) DO
        	value[j] := NULL;
        	DEC(j);
      END;
      IF value[i] = ':' THEN                       (* translate logical name *)
        IF Translate(value,i) THEN
         (* do nothing more in either case *)
        END;
      END;
      n := StringToCard(value);
      IF Done() AND (n > 0) THEN
      (* return *)
      ELSE 
        WriteString('Bad /');
        WriteString(qualifier);
        WriteString(' value! (=');
        WriteString(value);
        Write(')');
        WriteLn;
        WriteString('Specify a positive integer.');
        WriteLn;
        Halt(2);
      END;
    ELSE 
      n := 0;                                      (* qualifier not present *)
    END;
  END GetCardinal;

(******************************************************************************)

  PROCEDURE GetPosDimension (qualifier : ARRAY OF CHAR;
                             VAR pixels : CARDINAL);

(* Check if qualifier is present.  If so, then get value, check it is
   a valid positive dimension, convert and return via pixels.
   A valid dimension consists of a positive integer or real number followed
   by a two-letter unit: IN, CM, MM, PC, PT or PX (or in lowercase).
*)

    VAR
      i, j, status : CARDINAL;
      r : REAL;
      ch1, ch2 : CHAR;
      units : (in,cm,mm,pc,pt,px);
      RetLen : SHORTWORD;

  BEGIN
    IF ODD(CLI$PRESENT(qualifier)) THEN
      (* CLD ensures it has a value *)
      status := CLI$GET_VALUE(qualifier,value,RetLen);
      i := CARDINAL(ShortToInt(RetLen)) - 1;
      j := HIGH(value);
      WHILE (j > i) DO				   (* remove trailing blanks *)
        	value[j] := NULL;		   (* SYSDEP: pad with NULLs *)
        	DEC(j);
      END;
      IF value[i] = ':' THEN                       (* translate logical name *)
        IF Translate(value,i) THEN
         (* do nothing more in either case *)
        END;
      END;
      IF i = 0 THEN
        i := 1
      END;
   (* extract units *)
      IF    (Cap(value[i-1]) = 'I') AND (Cap(value[i]) = 'N') THEN
        units := in;
      ELSIF (Cap(value[i-1]) = 'C') AND (Cap(value[i]) = 'M') THEN
        units := cm;
      ELSIF (Cap(value[i-1]) = 'M') AND (Cap(value[i]) = 'M') THEN
        units := mm;
      ELSIF (Cap(value[i-1]) = 'P') AND (Cap(value[i]) = 'C') THEN
        units := pc;
      ELSIF (Cap(value[i-1]) = 'P') AND (Cap(value[i]) = 'T') THEN
        units := pt;
      ELSIF (Cap(value[i-1]) = 'P') AND (Cap(value[i]) = 'X') THEN
        units := px;
      ELSE 
        WriteString('Bad units in /');
        WriteString(qualifier);
        WriteString(' dimension! (=');
        WriteString(value);
        Write(')');
        WriteLn;
        WriteString('Last two letters should be IN, CM, MM, PC, PT or PX.');
        WriteLn;
        Halt(2);
      END;
      ch1 := value[i-1];             (* remember letters in units *)
      ch2 := value[i];
      value[i-1]   := NULL;            (* remove units *)
      value[i] := NULL;
      r := StringToReal(value);
      IF Done() AND (r > 0.0) THEN   (* convert r to pixels *)
        CASE units OF
          in : pixels := TRUNC(r * FLOAT(resolution) + 0.5);
        | cm : pixels := TRUNC((r / 2.54) * FLOAT(resolution) + 0.5);
        | mm : pixels := TRUNC((r / 25.4) * FLOAT(resolution) + 0.5);
        | pt : pixels := TRUNC((r / 72.27) * FLOAT(resolution) + 0.5);
        | pc : pixels := TRUNC((r / 72.27) * 12.0 * FLOAT(resolution) + 0.5);
        | px : pixels := TRUNC(r + 0.5);
        END;
      ELSE 
        value[i-1] := ch1;          (* restore units *)
        value[i]   := ch2;
        WriteString('Bad /');
        WriteString(qualifier);
        WriteString(' value! (=');
        WriteString(value);
        Write(')');
        WriteLn;
        WriteString('Specify a positive dimension.');
        WriteLn;
        Halt(2);
      END;
    ELSE 
      pixels := 0;                   (* qualifier not present *)
    END;
  END GetPosDimension;


(******************************************************************************)

  PROCEDURE GetDimension (qualifier : ARRAY OF CHAR;
                          VAR pixels : INTEGER);

(* Check if qualifier is present.  If so, then get value, check it is
   a valid dimension, convert and return via pixels.
   A valid dimension consists of a positive integer or real number followed
   by a two-letter unit: IN, CM, MM, PC, PT or PX (or in lowercase).
*)

    VAR
      i, j, status : CARDINAL;
      r : REAL;
      ch1, ch2 : CHAR;
      units : (in,cm,mm,pc,pt,px);
      RetLen : SHORTWORD;

  BEGIN
    IF ODD(CLI$PRESENT(qualifier)) THEN
      (* CLD ensures it has a value *)
      status := CLI$GET_VALUE(qualifier,value,RetLen);
      i := CARDINAL(ShortToInt(RetLen)) - 1;
      j := HIGH(value);
      WHILE (j > i) DO				   (* remove trailing blanks *)
        	value[j] := NULL;		   (* SYSDEP: pad with NULLs *)
        	DEC(j);
      END;
      IF value[i] = ':' THEN                       (* translate logical name *)
        IF Translate(value,i) THEN
         (* do nothing more in either case *)
        END;
      END;
      IF i = 0 THEN
        i := 1
      END;
   (* extract units *)
      IF    (Cap(value[i-1]) = 'I') AND (Cap(value[i]) = 'N') THEN
        units := in;
      ELSIF (Cap(value[i-1]) = 'C') AND (Cap(value[i]) = 'M') THEN
        units := cm;
      ELSIF (Cap(value[i-1]) = 'M') AND (Cap(value[i]) = 'M') THEN
        units := mm;
      ELSIF (Cap(value[i-1]) = 'P') AND (Cap(value[i]) = 'C') THEN
        units := pc;
      ELSIF (Cap(value[i-1]) = 'P') AND (Cap(value[i]) = 'T') THEN
        units := pt;
      ELSIF (Cap(value[i-1]) = 'P') AND (Cap(value[i]) = 'X') THEN
        units := px;
      ELSE 
        WriteString('Bad units in /');
        WriteString(qualifier);
        WriteString(' dimension! (=');
        WriteString(value);
        Write(')');
        WriteLn;
        WriteString('Last two letters should be IN, CM, MM, PC, PT or PX.');
        WriteLn;
        Halt(2);
      END;
      ch1 := value[i-1];             (* remember letters in units *)
      ch2 := value[i];
      value[i-1]   := NULL;            (* remove units *)
      value[i] := NULL;
      r := StringToReal(value);
      IF Done() THEN			(* convert r to pixels *)
        CASE units OF
          in : pixels := TRUNC(ABS(r) * FLOAT(resolution) + 0.5);
        | cm : pixels := TRUNC((ABS(r)/2.54) * FLOAT(resolution) + 0.5);
        | mm : pixels := TRUNC((ABS(r)/25.4) * FLOAT(resolution) + 0.5);
        | pt : pixels := TRUNC((ABS(r)/72.27) * FLOAT(resolution) + 0.5);
        | pc : pixels := TRUNC((ABS(r)/72.27) * 12.0 * FLOAT(resolution) + 0.5);
        | px : pixels := TRUNC(ABS(r) + 0.5);
        END;
        IF r < 0.0 THEN pixels := -pixels END;
      ELSE 
        value[i-1] := ch1;          (* restore units *)
        value[i]   := ch2;
        WriteString('Bad /');
        WriteString(qualifier);
        WriteString(' value! (=');
        WriteString(value);
        Write(')');
        WriteLn;
        WriteString('Specify a valid dimension.');
        WriteLn;
        Halt(2);
      END;
    ELSE 
      pixels := 0;                   (* qualifier not present *)
    END;
  END GetDimension;

(******************************************************************************)

  PROCEDURE Cap (ch : CHAR) : CHAR;

(* Hamburg's CAP is stupid - do my own. *)

  BEGIN
    IF (ch < 'a') OR (ch > 'z') THEN
      RETURN ch;
    ELSE 
      RETURN CHR(ORD(ch) - 32);
    END;
  END Cap;

(******************************************************************************)

  PROCEDURE GetString (qualifier : ARRAY OF CHAR;
                       VAR s     : stringvalue;
		       VAR untranslated : stringvalue);

(* Check if qualifier is present.  If so, then get value and return via s. *)

    VAR
      i, j, status : CARDINAL;
      RetLen : SHORTWORD;

  BEGIN
    IF ODD(CLI$PRESENT(qualifier)) THEN
      (* CLD ensures it has a value *)
      status := CLI$GET_VALUE(qualifier,s,RetLen);
      (* Check for the NULL string *)
      IF ShortToInt(RetLen) > 0 THEN
	i := CARDINAL(ShortToInt(RetLen)) - 1;
	j := HIGH(s);
	WHILE (j > i) DO
		  s[j] := NULL;
		  DEC(j);
	END;
	untranslated := s;	(* Copy untranslated qualifier value *)
	IF s[i] = ':' THEN                 (* translate logical name *)
	  IF Translate(s,i) THEN
	   (* do nothing more in either case *)
	  END;
	END;
      ELSE
	s[0] := NULL;
	untranslated := s;	(* Copy qualifier value *)
      END;
    ELSE 
      s[0] := NULL;
      untranslated := s;	(* Copy qualifier value *)
      (* SYSDEP: LEN(s) will be 0 *)
    END;
(* the main module will detect bad s value sooner or later *)
  END GetString;

(******************************************************************************)

  PROCEDURE Append (VAR s1 : ARRAY OF CHAR; s2 : ARRAY OF CHAR);

(* Append s2 to s1. *)

    VAR
      i, j : CARDINAL;

  BEGIN
    i := LEN(s1);   (* SYSDEP: assumes s1 ends with NULL, unless full *)
    j := 0;
    WHILE (i <= HIGH(s1)) AND (j <= HIGH(s2)) AND (s2[j] <> NULL) DO
      s1[i] := s2[j];
      INC(i);
      INC(j);
    END;
(* check for overflow??? *)
(* DEBUG
IF (i > HIGH(s1)) AND (j <= HIGH(s2)) AND (s2[j] <> NULL) THEN
   WriteString('No room to append '); WriteString(s2); WriteLn;
   Halt(2);
END;
GUBED *)
    IF i <= HIGH(s1) THEN
      s1[i] := NULL
    END;
  END Append;

(******************************************************************************)

(* SYSDEP: CLD file is used to supply most qualifiers with default values. *)

BEGIN
  GetDVIFile;                             (* initialize DVIname *)
  GetCardinal('MAGNIFICATION',mag);       (* 0 if no /MAG override *)
  GetCardinal('RESOLUTION',resolution);   (* get resolution BEFORE dimens *)
  GetPosDimension('XSIZE',paperwd);
  GetPosDimension('YSIZE',paperht);
  GetDimension('HOFFSET',hoffset);        (* 0 if not given *)
  GetDimension('VOFFSET',voffset);        (* 0 if not given *)
  GetDimension('XORIGIN',xorigin);	  (* default of 1" set by .CLD file *)
  GetDimension('YORIGIN',yorigin);	  (* default of 1" set by .CLD file *)
  GetString('VDU',vdu,DiscardValue);
  GetString('HELP_FILE',helpname,DiscardValue);
  GetString('PSPREFIX',psprefix,DiscardValue);
  GetString('TFM_DIRECTORY',DiscardValue,tfmdir); (* Use untranslated only *)
  GetString('PK_FONT_DIRECTORY',PKfontdir,PKfontPfx);
  GetString('PXL_FONT_DIRECTORY',PXLfontdir,PXLfontPfx);
  GetString('DUMMY_FONT',dummyfont,DiscardValue);
END DCLInterface.
