/***************************************************************************
** Although considerable effort has been expended to make this software   **
** correct and reliable, no warranty is implied; the author disclaims any **
** obligation or liability for damages, including but not limited to      **
** special, indirect, or consequential damages arising out of or in       **
** connection with the use or performance of this software.               **
***************************************************************************/

/*
 *	This file contains routines for manipulating Heaps. This
 *	data structure seems to apply best for this application; that is,
 *	insertion of random data at (mostly) random times, then extraction
 *	in sorted order.
 */

#include "debug.h"

/*
 *
 *	A Heap Node:
 */

struct Heap_Node {
                 long Key_Value;	/* Sort key */
	unsigned long User_Value;	/* User supplied node value */
};

/*
 *	The Heap Header is a special structure used for controlling the
 *	Heap:
 */

struct Heap_Head {
	unsigned int Next_Insert;	/* Index of next slot for insertion */
	unsigned int Max_Index;		/* Maximum number of nodes */
	         int Heap_Type;		/* -1: Descending; +1: Ascending */
	struct Heap_Node Heap[];	/* The collection of nodes */
};

/*
 *	These macros define how to get around in the Heap:
 */

#define Parent(k)      (k>>1)
#define Left_Child(k)  (k<<1)
#define Right_Child(k) ((k<<1)+1)
#define Is_Left(k)     ((k&0x01)==0)
#define Is_Right(k)    ((k&0x01)!=0)
#define Is_Root(k)     (k==1)

/*
 *	Routine Initialize_Heap creates a new Heap and initializes it
 *	for insertion. Note that node 0 is not used to make the
 *	parent-child relationship simpler.
 */

struct Heap_Head *Initialize_Heap (Max_Nodes, Type)
unsigned int Max_Nodes;
int Type;
{
	auto   struct Heap_Head *Heap_Ptr;
	extern char *Mem_Alloc();

	Heap_Ptr = (struct Heap_Head *) Mem_Alloc (sizeof (struct Heap_Head) +
			(Max_Nodes + 1) * sizeof (struct Heap_Node));
	Heap_Ptr->Next_Insert = 1;
	Heap_Ptr->Max_Index = Max_Nodes + 1;
	Heap_Ptr->Heap_Type = (Type >= 0) ? 1 : -1;
	return (Heap_Ptr);
}

/*
 *	Routine Insert_Heap_Entry places a new entry into the heap. It
 *	returns the number of nodes in the heap or zero if the new
 *	entry will not fit. Node zero is used to store the data for
 *	convenience while the process is underway.
 */

int Insert_Heap_Entry (Heap_Pointer, Key_Val, User_Val)
struct Heap_Head *Heap_Pointer;
long Key_Val;
unsigned long User_Val;
{
	auto   struct Heap_Head *Heap_Ptr;
	auto   unsigned int Index_1, Index_2;
	extern int Compare_Heap_Nodes();

	Heap_Ptr = Heap_Pointer;
	if (Heap_Ptr->Next_Insert >= Heap_Ptr->Max_Index)
		return (0);
	Heap_Ptr->Heap[0].Key_Value = Key_Val;
	Heap_Ptr->Heap[0].User_Value = User_Val;
	Index_1 = Heap_Ptr->Next_Insert++;
	while (!Is_Root(Index_1)) {
		Index_2 = Parent(Index_1);
		if (Compare_Heap_Nodes (Heap_Ptr, 0, Index_2) < 0) {
			Heap_Ptr->Heap[Index_1] = Heap_Ptr->Heap[Index_2];
			Index_1 = Index_2;
		} else break;
	}
	Heap_Ptr->Heap[Index_1] = Heap_Ptr->Heap[0];
	return (Heap_Ptr->Next_Insert-1);
}

/*
 *	Routine Extract_Heap_Entry extracts the top entry from the Heap
 *	and Re-Heapifies the Heap. The function returns a zero value when
 *	there are no more entries in the heap.
 */

int Extract_Heap_Entry (Heap_Pointer, Key_Value_Ptr, User_Value_Ptr)
struct Heap_Head *Heap_Pointer;
long *Key_Value_Ptr;
unsigned long *User_Value_Ptr;
{
	auto   unsigned int Index;
	auto   struct Heap_Head *Heap_Ptr;

	Heap_Ptr = Heap_Pointer;
	if (Is_Root(Heap_Ptr->Next_Insert))
		return (0);
	*Key_Value_Ptr = Heap_Ptr->Heap[1].Key_Value;
	*User_Value_Ptr = Heap_Ptr->Heap[1].User_Value;
	Index = --Heap_Ptr->Next_Insert;
	Heap_Ptr->Heap[1] = Heap_Ptr->Heap[Index];
	Adjust_Heap (Heap_Ptr, 1);
	return (1);
}

/*
 *	Routine Adjust_Heap re-heapifies a heap with only its
 *	root node out-of-order:
 */

Adjust_Heap (Heap_Pointer, Root_Index)
struct Heap_Head *Heap_Pointer;
unsigned int Root_Index;
{
	auto   struct Heap_Head *Heap_Ptr;
	auto   unsigned int Index_1, Index_2;
	extern int Compare_Heap_Nodes();

	Heap_Ptr = Heap_Pointer;
	Index_1 = Root_Index;
	Heap_Ptr->Heap[0] = Heap_Ptr->Heap[Index_1];
	while ((Index_2 = Left_Child(Index_1)) < Heap_Ptr->Next_Insert) {
		if (Index_2+1 < Heap_Ptr->Next_Insert &&
		    Compare_Heap_Nodes (Heap_Ptr, Index_2, Index_2+1) > 0)
			Index_2++;	/* Right Child is smaller */
		if (Compare_Heap_Nodes (Heap_Ptr, 0, Index_2) > 0) {
			Heap_Ptr->Heap[Index_1] = Heap_Ptr->Heap[Index_2];
			Index_1 = Index_2;
		} else break;
	}
	Heap_Ptr->Heap[Index_1] = Heap_Ptr->Heap[0];
}

/*
 *	Routine Compare_Heap_Nodes compares two heap nodes and returns
 *	-1 if the first node belongs on top of the second node; 0 if
 *	the nodes are equal and +1 if the first node belongs below the
 *	second node:
 */

int Compare_Heap_Nodes (Heap_Pointer, Index_1, Index_2)
struct Heap_Head *Heap_Pointer;
unsigned int Index_1, Index_2;
{
	auto   struct Heap_Head *Heap_Ptr;
	auto   struct Heap_Node *Heap_Node_Ptr_1, *Heap_Node_Ptr_2;
	auto   int H_Type, Cond;

	Heap_Ptr = Heap_Pointer;
	H_Type = Heap_Ptr->Heap_Type;
	Heap_Node_Ptr_1 = &Heap_Ptr->Heap[Index_1];
	Heap_Node_Ptr_2 = &Heap_Ptr->Heap[Index_2];
	if (Heap_Node_Ptr_1->Key_Value == Heap_Node_Ptr_2->Key_Value)
		Cond = 0;
	else if ((H_Type > 0 && Heap_Node_Ptr_1->Key_Value < Heap_Node_Ptr_2->Key_Value) ||
		 (H_Type < 0 && Heap_Node_Ptr_1->Key_Value > Heap_Node_Ptr_2->Key_Value))
		Cond = -1;
	else
		Cond = 1;
	return (Cond);
}

/*
 *	Routine Dissolve_Heap gets rid of the heap.
 */

Dissolve_Heap (Heap_Ptr)
struct Heap_Head *Heap_Ptr;
{
	Mem_Free (Heap_Ptr);
}

#ifdef DEBUG_HEAP
Dump_Heap (Heap_Ptr, Index)
struct Heap_Head *Heap_Ptr;
unsigned int Index;
{
	if (Index < Heap_Ptr->Next_Insert) {
		printf ("Heap Index %u:\n", Index);
		printf ("  Key Value is  %d\n", Heap_Ptr->Heap[Index].Key_Value);
		printf ("  User Value is %u\n", Heap_Ptr->Heap[Index].User_Value);
		Dump_Heap (Heap_Ptr, Left_Child(Index));
		Dump_Heap (Heap_Ptr, Right_Child(Index));
	}
}
#endif
