/***************************************************************************
** Although considerable effort has been expended to make this software   **
** correct and reliable, no warranty is implied; the author disclaims any **
** obligation or liability for damages, including but not limited to      **
** special, indirect, or consequential damages arising out of or in       **
** connection with the use or performance of this software.               **
***************************************************************************/

/*
 *	This file contains routines for non-host-specific file
 *	access capabilities.
 */

#include "types.h"

#include "fio.p"
#include "lbrio.p"

struct GFCB {
	pointer Lib_File_Desc;
	pointer File_Desc;
};

/*
 *	Routine Open_Gen_File opens either a normal file or a module
 *	in a library file. For the latter, the syntax of the file
 *	specification is the library name with the module name in
 *	parenthesis, e.g.:
 *
 *		libname(modname)
 *
 *	where both the 'libname' and 'modname' must adhere to host
 *	computer system syntax. In the case of a library/module being
 *	opened, the default name is not used.
 */

struct GFCB *Open_Gen_File (Name, Default_Name, Mode, EOLStr)
char *Name, *Default_Name, *Mode, *EOLStr;
{
	auto   struct GFCB *Gfcb_Ptr;
	auto   char *Str_Ptr, *Module_Ptr, *Ptr;
	static unsigned long Status[2];
	extern char *Mem_Alloc();
	extern int strlen();
/*
 *	Make a copy of the input name:
 */
	Str_Ptr = Mem_Alloc (strlen (Name) + 1);
	strcpy (Str_Ptr, Name);
/*
 *	Parse the file name string; determine if a library file specification
 *	or a normal file specification has been given:
 */
	for (Module_Ptr = Str_Ptr; *Module_Ptr != '\0'; Module_Ptr++)
	if (*Module_Ptr == '(') {
		*Module_Ptr++ = '\0';
		for (Ptr = Module_Ptr; *Ptr != ')' && *Ptr != '\0'; Ptr++)
			;
		*Ptr = '\0';
		break;
	}
/*
 *	Allocate a descriptor:
 */
	Gfcb_Ptr = (struct GFCB *) Mem_Alloc (sizeof (struct GFCB));
	Gfcb_Ptr->Lib_File_Desc = 0;
	Gfcb_Ptr->File_Desc = 0;
/*
 *	If there was no module specified, or if it was null, open
 *	a regular file; else open the library file and the module
 *	within the library:
 */
	if (*Module_Ptr == '\0')
		Gfcb_Ptr->File_Desc = Open_File_M (Str_Ptr, Default_Name, Mode, EOLStr);
	else {
		Gfcb_Ptr->Lib_File_Desc = Open_Text_Library_M (Str_Ptr, 0, Mode, Status);
		if (Gfcb_Ptr->Lib_File_Desc != 0)
			Gfcb_Ptr->File_Desc = Open_Library_File_M (Module_Ptr, EOLStr, Gfcb_Ptr->Lib_File_Desc,
								   Status);
	}
/*
 *	Check for an open failure:
 */
	if (Gfcb_Ptr->File_Desc == 0) {
		if (Gfcb_Ptr->Lib_File_Desc != 0)
			Close_Library_M (Gfcb_Ptr->Lib_File_Desc);
		Mem_Free (Gfcb_Ptr);
		Gfcb_Ptr = 0;
	}
	Mem_Free (Str_Ptr);
	return (Gfcb_Ptr);
}

Close_Gen_File (Gfcb_Ptr)
struct GFCB *Gfcb_Ptr;
{
	if (Gfcb_Ptr->Lib_File_Desc != 0) {
		Close_Library_File_M (Gfcb_Ptr->File_Desc);
		Close_Library_M (Gfcb_Ptr->Lib_File_Desc);
	} else
		Close_File_M (Gfcb_Ptr->File_Desc);
	Mem_Free (Gfcb_Ptr);
}

/*
 *	Routine Read_Gen_Character reads a character from the file
 *	opened by Open_Gen_File:
 */

unsigned char Read_Gen_Character (Gfcb_Ptr)
struct GFCB *Gfcb_Ptr;
{
	return ((Gfcb_Ptr->Lib_File_Desc != 0) ? Read_Library_Character_M (Gfcb_Ptr->File_Desc) :
						 Read_Character_M (Gfcb_Ptr->File_Desc));
}

int Gen_At_EOF (Gfcb_Ptr)
struct GFCB *Gfcb_Ptr;
{
	return ((Gfcb_Ptr->Lib_File_Desc != 0) ? Lib_At_EOF_M (Gfcb_Ptr->File_Desc) :
						 File_At_EOF_M (Gfcb_Ptr->File_Desc));
}

/*
 *	The following routines read a variety of entities from an
 *	input file.
 */

long Read_Signed_Int (Size, File)
unsigned int Size;
pointer File;
{
	auto   int Index;
	auto   unsigned long Long_Value;
	static unsigned char Byte_Value[4];
	static unsigned long Negation_Mask[4] =
		{ 0xFFFFFFFF, 0xFFFFFF00, 0xFFFF0000, 0xFF000000 };

	if (Size == 0 || Read_Block_M (Byte_Value, Size, File) != Size)
		return (0);
	Long_Value = 0;
	for (Index = 0; Index < Size; Index++)
		Long_Value = (Long_Value << 8) | Byte_Value[Index];
	if (Size < 4 && (Byte_Value[0] & 0x80) != 0)	/* Negative */
		Long_Value |= Negation_Mask[Size];
	return ((long) Long_Value);
}

unsigned long Read_Unsigned_Int (Size, File)
unsigned int Size;
pointer File;
{
	auto   int Index;
	auto   unsigned long Long_Value;
	static unsigned char Byte_Value[4];

	if (Size == 0 || Read_Block_M (Byte_Value, Size, File) != Size)
		return (0);
	Long_Value = 0;
	for (Index = 0; Index < Size; Index++)
		Long_Value = (Long_Value << 8) | Byte_Value[Index];
	return (Long_Value);
}

long Read_Signed_Int_Reverse (Size, File)
unsigned int Size;
pointer File;
{
	auto   int Index;
	auto   unsigned long Long_Value;
	static unsigned long Negation_Mask[4] =
		{ 0xFFFFFFFF, 0xFFFFFF00, 0xFFFF0000, 0xFF000000 };
	static unsigned long Sign_Mask[4] =
		{ 0x00000000, 0x00000080, 0x00008000, 0x00800000 };

	Long_Value = 0;
	for (Index = 0; Index < Size; Index++)
		Long_Value |= ((unsigned long) Read_Character_Reverse_M (File) <<
			       (Index << 3));
	if (Size < 4 && (Long_Value & Sign_Mask[Size]) != 0)
		Long_Value |= Negation_Mask[Size];
	return ((long) Long_Value);
}

unsigned long Read_Unsigned_Int_Reverse (Size, File)
unsigned int Size;
pointer File;
{
	auto   int Index;
	auto   unsigned long Long_Value;

	Long_Value = 0;
	for (Index = 0; Index < Size; Index++)
		Long_Value |= ((unsigned long) Read_Character_Reverse_M (File) <<
			       (Index << 3));
	return (Long_Value);
}

/*
 *	The next two routines, Read_Nybble and Read_Bit, read from
 *	a file one nybble or one bit at a time. They are designed
 *	to be alternately callable, if necessary - 4 bits and one
 *	nybble can be read, one nybble and 4 bits, or 8 bits. If an
 *	intervening call to routine Read_Character is made, the context
 *	value must be reset to zero to work correctly.
 */

struct Read_Context {
	unsigned char Value;
	unsigned char Next_Field;
	unsigned : 16;
};

unsigned char Read_Nybble (Context_Ptr, File)
struct Read_Context *Context_Ptr;
pointer File;
{
	auto   struct Read_Context *Ctx_Ptr;
	auto   int Field;

	Ctx_Ptr = Context_Ptr;
	if ((Field = (int) Ctx_Ptr->Next_Field) == 0 || Field > 4) {
		Ctx_Ptr->Value = Read_Character_M (File);
		Ctx_Ptr->Next_Field = 4;
		return (Ctx_Ptr->Value >> 4);
	}
	Ctx_Ptr->Next_Field = 0;
	return (Ctx_Ptr->Value & 0x0F);
}

unsigned char Read_Bit (Context_Ptr, File)
struct Read_Context *Context_Ptr;
pointer File;
{
	auto   struct Read_Context *Ctx_Ptr;
	auto   int Field, Bit;

	Ctx_Ptr = Context_Ptr;
	if ((Field = (int) Ctx_Ptr->Next_Field) == 0)
		Ctx_Ptr->Value = Read_Character_M (File);
	if ((Bit = (int) Ctx_Ptr->Value & (0x80 >> Field)) != 0)
		Bit = 1;
	Ctx_Ptr->Next_Field = (unsigned char) ((Field + 1) & 0x07);
	return ((unsigned char) Bit);
}
